//=============================================================================
// RPGツクールMZ - VoiceCommand.js
//-----------------------------------------------------------------------------
// このプラグインは以下のプラグインを参考に作成されました。
//
// トリアコンタン 様
// SimpleVoice.js
//=============================================================================

/*:ja
 * @target MZ
 * @plugindesc ボイスを出力するためのコマンドが使用できるようになります。
 * @author emoriiin979
 *
 * @help VoiceCommand.js
 *
 * ボイス出力をコモンイベント等で指定できるようになります。
 *
 * 音声ファイルの採用優先順:
 *   1. ステートID、スイッチID、変数条件全てに一致するもの
 *   2. ステートID、スイッチID両方に一致するもの
 *   3. ステートID、変数条件両方に一致するもの
 *   4. ステートIDのみ一致するもの
 *   5. スイッチID、変数条件両方に一致するもの
 *   6. スイッチIDのみ一致するもの
 *   7. 変数条件のみ一致するもの
 *   8. 条件なし (ステートID、スイッチID、変数条件全て設定なし)
 *
 * @param voiceSettings
 * @text 音声リスト
 * @desc ステート、スイッチ、変数条件ごとに立ち絵を複数定義できます。
 * ※この項目は使用しません。
 *
 * @param damageVoices
 * @text ダメージ時
 * @desc 被ダメージ時に出力する音声を定義します。
 * @default []
 * @type struct<voices>[]
 * @parent voiceSettings
 *
 * @param turnEndVoices
 * @text ターン終了時
 * @desc ターン終了時のイベント中に出力する音声を定義します。
 * @default []
 * @type struct<voices>[]
 * @parent voiceSettings
 *
 * @param optionName
 * @text オプション名
 * @desc オプション画面に表示されるボイス音量の設定項目名称です。
 * @default ボイス 音量
 * @type string
 *
 * @param defaultValue
 * @text オプション初期値
 * @desc ボイス音量の初期値です。
 * @default 40
 * @min 0
 * @max 100
 * @type number
 * 
 * @command doVoice
 * @text ボイスの出力
 * @desc ボイスを出力します。
 * ステート、スイッチ、変数条件によって自動的にボイスが選択されます。
 *
 * @arg voiceType
 * @text ボイス種類
 * @desc 出力するボイスの種類を選択してください。
 * @default damage
 * @type select
 * @option 被ダメージ
 * @value damage
 * @option 絶頂
 * @value orgasm
 *
 * @command forceVoice
 * @text ボイスの指定出力...
 * @desc 指定のボイスを出力します。
 *
 * @arg voiceName
 * @text ボイス名
 * @desc 出力する音声ファイルを選択してください。
 * @dir audio/voice
 * @type file
 * @require 1
 *
 * @command playBGVoice
 * @text BGボイスの指定出力...
 * @desc 指定のBGボイスを出力します。
 *
 * @arg voiceName
 * @text ボイス名
 * @desc 出力する音声ファイルを選択してください。
 * @dir audio/bgv
 * @type file
 * @require 1
 *
 * @command fadeOutBGVoice
 * @text BGボイスのフェードアウト
 * @desc BGボイスをフェードアウトします。
 *
 * @arg duration
 * @text 時間
 * @desc フェードアウトにかかる時間を指定します。
 * @default 1
 * @type number
 */

/*~struct~voices:
 *
 * @param memo
 * @text メモ
 * @desc この項目は使用しません。
 * メモとしてご利用ください。
 *
 * @param voiceName
 * @text 音声ファイル名
 * @desc 出力する音声ファイルを選択してください。
 * @dir audio/voice
 * @type file
 * @require 1
 *
 * @param actorId
 * @text アクターID
 * @desc アクターIDです。音声を定義するアクターを選択してください。
 * @default 1
 * @type actor
 *
 * @param stateCase
 * @text ステート条件
 * @desc ステート条件でボイスを変更したい場合に使用します。
 * @default []
 * @type struct<stateCase>[]
 *
 * @param switchCase
 * @text スイッチ条件
 * @desc スイッチ条件でボイスを変更したい場合に使用します。
 * @default []
 * @type struct<switchCase>[]
 *
 * @param variableCase
 * @text 変数条件
 * @desc 変数条件でボイスを変更したい場合に使用します。
 * @default []
 * @type struct<variableCase>[]
 */

/*~struct~stateCase:
 *
 * @param id
 * @text ステートID
 * @desc 条件に使用するステートIDです。
 * @type state
 */

/*~struct~switchCase:
 *
 * @param id
 * @text スイッチID
 * @desc 条件に使用するスイッチIDです。
 * @type switch
 */

/*~struct~variableCase:
 *
 * @param id
 * @text 変数ID
 * @desc 条件に使用する変数IDです。
 * @type variable
 *
 * @param type
 * @text 変数条件
 * @desc 変数IDとの比較条件です。
 * @default equal
 * @type select
 * @option 一致
 * @value equal
 * @option 以上
 * @value higher
 * @option 以下
 * @value lower
 *
 * @param value
 * @text 変数比較数値
 * @desc 変数IDと比較する数値です。
 * @default 0
 * @min -99999999
 * @max 99999999
 * @type number
 */

(() => {
    "use strict";

    const pluginName = "VoiceCommand";
    const parameters = PluginManager.parameters(pluginName);

    //=========================================================================
    // プラグインコマンド
    //=========================================================================

    // ボイス出力
    PluginManager.registerCommand(pluginName, "doVoice", args => {
        AudioManager.playOptimalVoice(args["voiceType"]);
    });

    // ボイス指定出力
    PluginManager.registerCommand(pluginName, "forceVoice", args => {
        AudioManager.playVoice({
            name: args["voiceName"],
            pan: 0,
            pitch: 100,
            volume: 90,
        });
    });

    // BGボイス指定出力
    PluginManager.registerCommand(pluginName, "playBGVoice", args => {
        AudioManager.playBgv({
            name: args["voiceName"],
            pan: 0,
            pitch: 100,
            volume: 70,
        });
    });

    // BGボイスのフェードアウト
    PluginManager.registerCommand(pluginName, "fadeOutBGVoice", args => {
        AudioManager.fadeOutBgv(args["duration"]);
    });

    //-------------------------------------------------------------------------
    // ConfigManager
    //
    // Voice音量の設定項目を追加します。

    Object.defineProperty(ConfigManager, "voiceVolume", {
        get: function() {
            return AudioManager._voiceVolume;
        },
        set: function(value) {
            AudioManager.voiceVolume = value;
        },
        configurable: true,
    });

    const _ConfigManager_makeData = ConfigManager.makeData;
    ConfigManager.makeData = function() {
        const config = _ConfigManager_makeData.apply(this, arguments);
        config.voiceVolume = this.voiceVolume;
        return config;
    };

    //-------------------------------------------------------------------------
    // Window_Options
    //
    // ボイス音量の設定項目を追加します。

    const _Window_Option_addVolumeOptions = Window_Options.prototype.addVolumeOptions;
    Window_Options.prototype.addVolumeOptions = function() {
        _Window_Option_addVolumeOptions.apply(this, arguments);
        this.addCommand(parameters.optionName, "voiceVolume");
    };

    //-------------------------------------------------------------------------
    // Scene_Options
    //
    // オプションウィンドウの高さを1増やします。

    const _Scene_Options_maxCommands = Scene_Options.prototype.maxCommands;
    Scene_Options.prototype.maxCommands = function() {
        let count = _Scene_Options_maxCommands.apply(this, arguments);
        return ++count;
    };

    //-------------------------------------------------------------------------
    // AudioManager
    //
    // ボイスの演奏機能を追加定義します。

    Object.defineProperty(AudioManager, "voiceVolume", {
        get: function() {
            return this._voiceVolume;
        },
        set: function(value) {
            this._voiceVolume = value;
        },
        configurable: true,
    });

    AudioManager._voiceBuffers = [];
    AudioManager._voiceVolume = 100;

    AudioManager.playVoice = function(voice) {
        if (voice.name) {
            this.stopVoice();
            const buffer = this.createBuffer("voice/", voice.name);
            this.updateVoiceParameters(buffer, voice);
            buffer.play(false);
            this._voiceBuffers.push(buffer);
        }
    };

    AudioManager.updateVoiceParameters = function(buffer, voice) {
        this.updateBufferParameters(buffer, this._voiceVolume, voice);
    };

    AudioManager.stopVoice = function() {
        for (const buffer of this._voiceBuffers) {
            if (!buffer.isPlaying()) {
                buffer.destroy();
            }
        }
        this._voiceBuffers = [];
    };

    AudioManager.playOptimalVoice = function(voiceType) {
        let result = null;
        switch (voiceType) {
            case "damage":
                result = damageVoices.search();
                break;
            case "orgasm":
                result = turnEndVoices.search();
                break;
            default:
                // nop.
        }
        if (result) {
            AudioManager.playVoice({
                name: result.voiceName,
                pan: 0,
                pitch: 100,
                volume: 90,
            });
        }
    };

    AudioManager._currentBgv = null;
    AudioManager._bgvBuffer = null;
    // Note: 音量は `this._voiceVolume` を使いまわす

    AudioManager.playBgv = function(bgv, pos) {
        if (this.isCurrentBgv(bgv)) {
            this.updateBgvParameters(bgv);
        } else {
            if (bgv.name) {
                this.stopBgv();
                this._bgvBuffer = this.createBuffer("bgv/", bgv.name);
                this.updateBgvParameters(bgv);
                this._bgvBuffer.play(true, pos || 0);
            }
        }
        this.updateCurrentBgv(bgv, pos);
    };

    AudioManager.isCurrentBgv = function(bgv) {
        return (
            this._currentBgv &&
            this._bgvBuffer &&
            this._currentBgv.name === bgv.name
        );
    };

    AudioManager.updateBgvParameters = function(bgv) {
        this.updateBufferParameters(this._bgvBuffer, this._voiceVolume, bgv);
    };

    AudioManager.updateCurrentBgv = function(bgv, pos) {
        this._currentBgv = {
            name: bgv.name,
            volume: bgv.volume,
            pitch: bgv.pitch,
            pan: bgv.pan,
            pos: pos,
        };
    };

    AudioManager.stopBgv = function() {
        if (this._bgvBuffer) {
            this._bgvBuffer.destroy();
            this._bgvBuffer = null;
            this._currentBgv = null;
        }
    };

    AudioManager.fadeOutBgv = function(duration) {
        if (this._bgvBuffer && this._currentBgv) {
            this._bgvBuffer.fadeOut(duration);
            this._currentBgv = null;
        }
    };

    //-------------------------------------------------------------------------
    // Scene_Base
    //
    // フェードアウト時にVoiceの演奏も停止します。

    const _Scene_Base_fadeOutAll = Scene_Base.prototype.fadeOutAll;
    Scene_Base.prototype.fadeOutAll = function() {
        _Scene_Base_fadeOutAll.apply(this, arguments);
        AudioManager.stopVoice();
    }

    //-------------------------------------------------------------------------
    // VoiceList
    //
    // 音声リストクラスを追加定義します。

    function VoiceList() {
        this.initialize(...arguments);
    };

    VoiceList.prototype.initialize = function(voices) {
        this.voices = voices;
        this.randomId = 0;
    };

    VoiceList.prototype.search = function() {
        const actorId = 1;
        const actorStates = $gameActors.actor(actorId)._states;

        const matchStateCase = function(voice) {
            const res = voice.stateCase.filter((x) => {
                return actorStates.indexOf(Number(x.id)) !== -1;
            });
            return res.length > 0 && res.length === voice.stateCase.length;
        };
        const matchSwitchCase = function(voice) {
            const res = voice.switchCase.filter((x) => {
                return $gameSwitches.value(Number(x.id));
            });
            return res.length > 0 && res.length === voice.switchCase.length;
        };
        const matchVariableCase = function(voice) {
            const res = voice.variableCase.filter((x) => {
                return String(x.type) === "equal"
                        && $gameVariables.value(Number(x.id)) == Number(x.value)
                    || String(x.type) === "higher"
                        && $gameVariables.value(Number(x.id)) >= Number(x.value)
                    || String(x.type) === "lower"
                        && $gameVariables.value(Number(x.id)) <= Number(x.value);
            });
            return res.length > 0 && res.length === voice.variableCase.length;
        };

        const randomSelect = function(voices) {
            if (voices.length === 1) {
                this.randomId = 0;
                return voices[0];
            }
            const candidateIds = [];
            for (let i = 1; i <= voices.length; i++) {
                if (i !== this.randomId) {
                    candidateIds.push(i);
                }
            }
            this.randomId = candidateIds[Math.floor(Math.random() * candidateIds.length)];
            return voices[this.randomId - 1];
        }.bind(this);

        let filteredVoices = [];

        // 1. ステートID、スイッチID、変数条件全てに一致するもの
        filteredVoices = this.voices.filter(function(voice) {
            return matchStateCase(voice)
                && matchSwitchCase(voice)
                && matchVariableCase(voice);
        });
        if (filteredVoices.length) return randomSelect(filteredVoices);
    
        // 2. ステートID、スイッチID両方に一致するもの
        filteredVoices = this.voices.filter(function(voice) {
            return matchStateCase(voice)
                && matchSwitchCase(voice)
                && voice.variableCase.length === 0;
        });
        if (filteredVoices.length) return randomSelect(filteredVoices);
    
        // 3. ステートID、変数条件両方に一致するもの
        filteredVoices = this.voices.filter(function(voice) {
            return matchStateCase(voice)
                && voice.switchCase.length === 0
                && matchVariableCase(voice);
        });
        if (filteredVoices.length) return randomSelect(filteredVoices);
    
        // 4. ステートIDのみ一致するもの
        filteredVoices = this.voices.filter(function(voice) {
            return matchStateCase(voice)
                && voice.switchCase.length === 0
                && voice.variableCase.length === 0;
        });
        if (filteredVoices.length) return randomSelect(filteredVoices);
    
        // 5. スイッチID、変数条件両方に一致するもの
        filteredVoices = this.voices.filter(function(voice) {
            return voice.stateCase.length === 0
                && matchSwitchCase(voice)
                && matchVariableCase(voice);
        });
        if (filteredVoices.length) return randomSelect(filteredVoices);
    
        // 6. スイッチIDのみ一致するもの
        filteredVoices = this.voices.filter(function(voice) {
            return voice.stateCase.length === 0
                && matchSwitchCase(voice)
                && voice.variableCase.length === 0;
        });
        if (filteredVoices.length) return randomSelect(filteredVoices);
    
        // 7. 変数条件のみ一致するもの
        filteredVoices = this.voices.filter(function(voice) {
            return voice.stateCase.length === 0
                && voice.switchCase.length === 0
                && matchVariableCase(voice);
        });
        if (filteredVoices.length) return randomSelect(filteredVoices);
    
        // 8. 条件なし (ステートID、スイッチID、変数条件全て設定なし)
        filteredVoices = this.voices.filter(function(voice) {
            return voice.stateCase.length === 0
                && voice.switchCase.length === 0
                && voice.variableCase.length === 0;
        });
        if (filteredVoices.length) return randomSelect(filteredVoices);
    };

    // 全立ち絵を格納したオブジェクトを作成
    const parseJsonParam = function(paramName) {
        return new VoiceList(JSON.parse(
            JSON.stringify(
                String(parameters[paramName] || "[]"),
                function (key, value) {
                    try {
                        return JSON.parse(value);
                    } catch (e) {
                        return value;
                    }
                }
            )
        ));
    };

    const damageVoices = parseJsonParam("damageVoices");
    const turnEndVoices = parseJsonParam("turnEndVoices");
    
    //-------------------------------------------------------------------------
    // Game_Actor

    const _Game_Actor_performDamage = Game_Actor.prototype.performDamage;
    Game_Actor.prototype.performDamage = function() {
        _Game_Actor_performDamage.apply(this, arguments);
        AudioManager.playOptimalVoice("damage");
    };
})();
